/*
 * interfaces.c
 * Copyright (C) Aitor Cuadrado Zubizarreta <aitor_czr@gnuinos.org>
 * 
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * See the COPYING file. * 
 */

#include "interfaces.h"

void get_interfaces(struct sbuf* wired_interface, struct sbuf* wireless_interface)
{
    struct if_nameindex *if_nidxs, *intf;

    if_nidxs = if_nameindex();
    if ( if_nidxs != NULL )
    {
        for (intf = if_nidxs; intf->if_index != 0 || intf->if_name != NULL; intf++)
        {
            struct sbuf fp;
            sbuf_init(&fp);
            sbuf_addstr(&fp, "/sys/class/net/");
            sbuf_addstr(&fp, intf->if_name);
            sbuf_addstr(&fp, "/device/vendor");
            
            int f=exists(fp.buf);
			switch (f)
			{
				case 0:  {							
							struct stat buf;
							struct sbuf dir;
							
							sbuf_init(&dir);
							sbuf_addstr(&dir, "/sys/class/net/");
							sbuf_addstr(&dir, intf->if_name);
							sbuf_addstr(&dir, "/wireless");
							
							if(stat(dir.buf, &buf) == -1) sbuf_addstr(wired_interface, intf->if_name);
							else if(buf.st_mode & S_IFDIR) sbuf_addstr(wireless_interface, intf->if_name);							
							free(dir.buf);
						 }
						 
						 break;
				
				case -1: break;
				
				case -2: printf ("Enable to open %d (%s)\n", errno, strerror(errno));
						 break;
				
				default: break;
			}
			
			free(fp.buf);
        }

        if_freenameindex(if_nidxs);
    }
}

short exists(char *file)
{
	int fd=open(file, O_RDONLY);
	if (fd<0) return (errno==ENOENT)?-1:-2;
	close(fd);
	return 0;
}

void interface_up (const char *if_name)
{
    struct ifreq ifr;
    int skfd;	  /* generic raw socket desc. */

	/* Create a channel to the NET kernel. */
	if((skfd = iw_sockets_open()) < 0)
    {
		perror("Enable to create a channel to the NET kernel.");
		exit(-1);
    }

    strncpy(ifr.ifr_name, if_name, IFNAMSIZ);

    if (ioctl(skfd, SIOCGIFFLAGS, &ifr) >= 0) {
		printf("\nActivating interface %s.\n\n", if_name);
        strncpy(ifr.ifr_name, if_name, IFNAMSIZ);
        ifr.ifr_flags |= (IFF_UP | IFF_RUNNING);
        ioctl(skfd, SIOCSIFFLAGS, &ifr);
    } else {
        printf("\nGetting flags for interface %s failed, not activating interface.\n", if_name);
    }
    
    /* Close the socket. */
	iw_sockets_close(skfd);
}

void interface_down (const char *if_name)
{
    struct ifreq ifr;
    int skfd;	  /* generic raw socket desc. */

	/* Create a channel to the NET kernel. */
	if((skfd = iw_sockets_open()) < 0)
    {
		perror("\nEnable to create a channel to the NET kernel.\n");
		exit(-1);
    }

    strncpy(ifr.ifr_name, if_name, IFNAMSIZ);

    if (ioctl(skfd, SIOCGIFFLAGS, &ifr) >= 0) {
		printf("\nTaking down interface %s.\n\n", if_name);
        strncpy(ifr.ifr_name, if_name, IFNAMSIZ);
        ifr.ifr_flags &= ~IFF_UP;
        ioctl(skfd, SIOCSIFFLAGS, &ifr);
    } else {
        printf("\nGetting flags for interface %s failed, not taking down interface.\n", if_name);
    }
    
    /* Close the socket. */
	iw_sockets_close(skfd);
}

int get_interface_status(const char *if_name)
{
	bool status;
    struct ifreq ifr;
    int skfd;	  /* generic raw socket desc. */

	/* Create a channel to the NET kernel. */
	if((skfd = iw_sockets_open()) < 0)
    {
		perror("\nEnable to create a channel to the NET kernel.\n");
		return(-1);
    }

    strncpy(ifr.ifr_name, if_name, IFNAMSIZ);    
 
	if (ifr.ifr_flags&IFF_UP) status = 1;
	else status = 0;
    
    /* Close the socket. */
	iw_sockets_close(skfd);
	
	return status;
}   
